<?php
require_once 'config.php';

class AttendanceSync {
    private $conn;
    
    public function __construct() {
        try {
            $this->conn = new PDO(
                "mysql:host=" . Config::DB_HOST . ";dbname=" . Config::DB_NAME,
                Config::DB_USER,
                Config::DB_PASS
            );
            $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch(PDOException $e) {
            die("Database connection failed: " . $e->getMessage());
        }
    }
    
    // Fetch users from your API
    public function getUsersFromAPI() {
        echo "<h3>📡 Fetching Users from API</h3>";
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::USER_API_URL,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_FOLLOWLOCATION => true,
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            echo "❌ API Error: HTTP $httpCode<br>";
            return false;
        }
        
        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            echo "❌ JSON decode error: " . json_last_error_msg() . "<br>";
            echo "Response sample: " . htmlspecialchars(substr($response, 0, 200)) . "<br>";
            return false;
        }
        
        echo "✅ API Success - Received data<br>";
        return $data;
    }
    
    // Sync users to database (and eventually to ZKTeco)
    public function syncUsers($users) {
        echo "<h3>👥 Syncing Users to Database</h3>";
        
        if (!is_array($users) || empty($users)) {
            echo "❌ No users data received<br>";
            return 0;
        }
        
        $synced = 0;
        foreach ($users as $user) {
            try {
                // Extract user data based on your API structure
                $user_id = $user['user_id'] ?? $user['id'] ?? 'unknown';
                $name = $user['name'] ?? $user['username'] ?? 'Unknown';
                $email = $user['email'] ?? '';
                $department = $user['department'] ?? '';
                
                // Insert or update user
                $stmt = $this->conn->prepare("
                    INSERT INTO users (user_id, name, email, department, sync_status) 
                    VALUES (?, ?, ?, ?, 'pending') 
                    ON DUPLICATE KEY UPDATE 
                    name = VALUES(name), 
                    email = VALUES(email), 
                    department = VALUES(department),
                    sync_status = 'pending'
                ");
                
                $stmt->execute([$user_id, $name, $email, $department]);
                $synced++;
                
                echo "✅ User: $name ($user_id)<br>";
                
            } catch (Exception $e) {
                echo "❌ Error syncing user: " . $e->getMessage() . "<br>";
            }
        }
        
        echo "✅ Synced $synced users to database<br>";
        return $synced;
    }
    
    // Calculate attendance status based on your rules
    public function calculateAttendanceStatus($checkInTime) {
        $checkIn = date('H:i:s', strtotime($checkInTime));
        
        if ($checkIn <= '09:00:00') return 'P'; // Present
        if ($checkIn <= '09:15:00') return 'L'; // Late
        if ($checkIn <= '10:50:00') return 'HD'; // Half Day
        return 'A'; // Absent
    }
    
    // Test attendance rules
    public function testAttendanceRules() {
        echo "<h3>⏰ Testing Attendance Rules</h3>";
        
        $testCases = [
            '08:30:00' => 'P',
            '09:05:00' => 'L',
            '09:30:00' => 'HD',
            '11:00:00' => 'A'
        ];
        
        foreach ($testCases as $time => $expected) {
            $status = $this->calculateAttendanceStatus("2024-01-01 $time");
            $result = $status === $expected ? '✅' : '❌';
            echo "$result $time → $status (Expected: $expected)<br>";
        }
    }
    
    // Main sync function
    public function runSync() {
        echo "<h1>🔄 Attendance System Sync</h1>";
        
        // Test database
        echo "✅ Database connected successfully<br><br>";
        
        // Test attendance rules
        $this->testAttendanceRules();
        echo "<br>";
        
        // Fetch users from API
        $users = $this->getUsersFromAPI();
        
        if ($users) {
            echo "User data type: " . gettype($users) . "<br>";
            echo "User data count: " . (is_array($users) ? count($users) : 'N/A') . "<br>";
            
            // Show sample of what we received
            if (is_array($users) && count($users) > 0) {
                echo "<h4>Sample User Data:</h4>";
                echo "<pre>" . print_r(array_slice($users, 0, 3), true) . "</pre>";
                
                // Sync users to database
                $syncedCount = $this->syncUsers($users);
                echo "<h3>🎉 Sync completed: $syncedCount users processed</h3>";
            }
        }
    }
}

// Run the sync
$sync = new AttendanceSync();
$sync->runSync();

echo "<hr><h3>Next Steps:</h3>";
echo "1. Check the API response format above<br>";
echo "2. Update ZKTeco device IP when ready<br>";
echo "3. Implement actual ZKTeco communication<br>";
?>