<?php
class AttendanceSystemSync {
    // Database configuration
    const DB_HOST = 'localhost';
    const DB_NAME = 'onesqeif_799';
    const DB_USER = 'onesqeif_799';
    const DB_PASS = '!QJ{xhY5Dkz1';
    
    // API Configuration
    const USER_API_URL = 'https://ams.onesbd.com/get_user.php';
    const ZK_TECHO_IP = '192.168.0.103';
    const ZK_TECHO_PORT = 4370;
    
    // Attendance Rules
    const START_TIME = '08:00:00';
    const LATE_THRESHOLD = '09:00:00';
    const LATE_CUTOFF = '09:15:00';
    const HALF_DAY_CUTOFF = '10:50:00';
    const CHECK_OUT_TIME = '12:00:00';

    private $pdo;

    public function __construct() {
        try {
            // Database connection
            $this->pdo = new PDO(
                "mysql:host=" . self::DB_HOST . ";dbname=" . self::DB_NAME, 
                self::DB_USER, 
                self::DB_PASS
            );
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            echo "✅ Database connected successfully\n";
        } catch (PDOException $e) {
            die("❌ Database connection failed: " . $e->getMessage());
        }
    }

    public function fixDatabaseSchema() {
        echo "🔧 Fixing Database Schema\n";
        
        try {
            // Check if type column exists in users table
            $stmt = $this->pdo->query("DESCRIBE users");
            $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            if (!in_array('type', $columns)) {
                echo "➡️ Adding missing 'type' column to users table...\n";
                $this->pdo->exec("ALTER TABLE users ADD COLUMN type ENUM('staff', 'student') NOT NULL DEFAULT 'student' AFTER phone");
                echo "✅ Added 'type' column to users table\n";
            }
            
            echo "✅ Database schema is correct\n";
        } catch (PDOException $e) {
            echo "❌ Schema fix error: " . $e->getMessage() . "\n";
        }
    }

    public function checkTables() {
        echo "🗃️ Checking Database Tables\n";
        
        $tables = ['users', 'zk_users', 'attendance', 'student_attendance'];
        foreach ($tables as $table) {
            try {
                $stmt = $this->pdo->query("SELECT COUNT(*) as count FROM $table");
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                echo "✅ $table: {$result['count']} records\n";
            } catch (PDOException $e) {
                echo "❌ $table: " . $e->getMessage() . "\n";
            }
        }
    }

    public function testAttendanceRules() {
        echo "⏰ Testing Attendance Rules\n";
        
        $testTimes = [
            '07:45:00' => 'P',
            '08:00:00' => 'P',
            '08:30:00' => 'P',
            '09:05:00' => 'L',
            '09:15:00' => 'L',
            '09:30:00' => 'HD',
            '10:00:00' => 'HD',
            '10:50:00' => 'HD',
            '11:00:00' => 'A',
            '12:00:00' => 'A'
        ];

        foreach ($testTimes as $time => $expected) {
            $status = $this->calculateAttendanceStatus($time);
            $result = $status === $expected ? '✅' : '❌';
            echo "$result $time → $status (Expected: $expected)\n";
        }
    }

    private function calculateAttendanceStatus($checkTime) {
        $check = strtotime($checkTime);
        $start = strtotime(self::START_TIME);
        $lateThreshold = strtotime(self::LATE_THRESHOLD);
        $lateCutoff = strtotime(self::LATE_CUTOFF);
        $halfDayCutoff = strtotime(self::HALF_DAY_CUTOFF);

        if ($check <= $lateThreshold) return 'P';
        if ($check <= $lateCutoff) return 'L';
        if ($check <= $halfDayCutoff) return 'HD';
        return 'A';
    }

    public function fetchUsersFromAPI() {
        echo "📡 Fetching Users from API: " . self::USER_API_URL . "\n";
        
        try {
            // Enhanced API request with better error handling
            $context = stream_context_create([
                'ssl' => [
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                ],
                'http' => [
                    'timeout' => 30,
                    'ignore_errors' => true
                ]
            ]);

            $response = file_get_contents(self::USER_API_URL, false, $context);
            
            if ($response === false) {
                throw new Exception("Failed to fetch data from API. Check URL and connectivity.");
            }

            // Debug: Show raw response
            echo "📋 Raw API Response: " . substr($response, 0, 500) . "...\n";

            $users = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("JSON decode error: " . json_last_error_msg());
            }

            if (!is_array($users)) {
                throw new Exception("API returned non-array data. Response: " . $response);
            }

            echo "✅ API Success - Received " . count($users) . " users\n";
            
            if (count($users) > 0) {
                echo "📊 First 3 Users Sample:\n";
                for ($i = 0; $i < min(3, count($users)); $i++) {
                    echo "User $i: " . print_r($users[$i], true) . "\n";
                }
            }

            return $users;
        } catch (Exception $e) {
            echo "❌ API Error: " . $e->getMessage() . "\n";
            return [];
        }
    }

    public function syncUsersToDatabase($users) {
        echo "👥 Syncing Users to Database\n";
        
        $successCount = 0;
        $errorCount = 0;

        if (empty($users)) {
            echo "⚠️ No users to sync\n";
            return ['success' => 0, 'errors' => 0];
        }

        foreach ($users as $index => $user) {
            try {
                // Debug: Show what we're processing
                echo "Processing user $index: " . print_r($user, true) . "\n";

                // Handle different API response formats
                if (isset($user['uid']) && isset($user['name'])) {
                    // Standard format
                    $uid = $user['uid'];
                    $name = $user['name'];
                    $email = $user['email'] ?? '';
                    $phone = $user['phone'] ?? '';
                    $type = isset($user['type']) ? $user['type'] : 'student';
                } elseif (isset($user['user_id']) && isset($user['user_name'])) {
                    // Alternative format
                    $uid = $user['user_id'];
                    $name = $user['user_name'];
                    $email = $user['user_email'] ?? '';
                    $phone = $user['user_phone'] ?? '';
                    $type = $user['user_type'] ?? 'student';
                } else {
                    // Try to extract from available data
                    $uid = $user['id'] ?? $user['ID'] ?? $index + 1;
                    $name = $user['name'] ?? $user['Name'] ?? $user['username'] ?? 'Unknown User ' . $uid;
                    $email = $user['email'] ?? $user['Email'] ?? '';
                    $phone = $user['phone'] ?? $user['Phone'] ?? $user['mobile'] ?? '';
                    $type = $user['type'] ?? $user['Type'] ?? 'student';
                }

                // Validate required fields
                if (empty($name) || $name === 'Unknown') {
                    echo "⚠️ Skipping invalid user data at index $index\n";
                    continue;
                }

                // Ensure type is valid
                $type = ($type === 'staff' || $type === 'teacher') ? 'staff' : 'student';

                // Check if user already exists
                $stmt = $this->pdo->prepare("SELECT id FROM users WHERE uid = ? AND type = ?");
                $stmt->execute([$uid, $type]);
                $existingUser = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($existingUser) {
                    // Update existing user
                    $stmt = $this->pdo->prepare("
                        UPDATE users SET 
                        name = ?, email = ?, phone = ?, updated_at = NOW() 
                        WHERE id = ?
                    ");
                    $stmt->execute([$name, $email, $phone, $existingUser['id']]);
                    echo "✅ Updated: $name (UID: $uid, Type: $type)\n";
                } else {
                    // Insert new user
                    $stmt = $this->pdo->prepare("
                        INSERT INTO users (uid, name, email, phone, type, created_at, updated_at) 
                        VALUES (?, ?, ?, ?, ?, NOW(), NOW())
                    ");
                    $stmt->execute([$uid, $name, $email, $phone, $type]);
                    echo "✅ Added: $name (UID: $uid, Type: $type)\n";
                }

                $successCount++;

            } catch (PDOException $e) {
                $errorCount++;
                $userName = $user['name'] ?? 'Unknown';
                echo "❌ Error syncing user '$userName': " . $e->getMessage() . "\n";
            }
        }

        echo "📊 Sync Results: $successCount successful, $errorCount errors\n";
        return ['success' => $successCount, 'errors' => $errorCount];
    }

    public function testAPIConnection() {
        echo "🔍 Testing API Connection\n";
        
        $testUrl = self::USER_API_URL;
        echo "Testing URL: $testUrl\n";
        
        // Test with cURL for better debugging
        if (function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $testUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_error($ch)) {
                echo "❌ cURL Error: " . curl_error($ch) . "\n";
            } else {
                echo "✅ HTTP Status: $httpCode\n";
                echo "📦 Response length: " . strlen($response) . " bytes\n";
                echo "📄 Response preview: " . substr($response, 0, 200) . "...\n";
            }
            
            curl_close($ch);
            return $response;
        }
        
        return null;
    }

    public function getExistingUsers() {
        echo "👥 Existing Users in Database:\n";
        
        try {
            $stmt = $this->pdo->query("
                SELECT id, uid, name, type, created_at 
                FROM users 
                ORDER BY id DESC 
                LIMIT 5
            ");
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (count($users) > 0) {
                foreach ($users as $user) {
                    echo "• {$user['name']} (UID: {$user['uid']}, Type: {$user['type']}) - {$user['created_at']}\n";
                }
            } else {
                echo "No users found in database\n";
            }
        } catch (PDOException $e) {
            echo "❌ Error reading users: " . $e->getMessage() . "\n";
        }
    }

    public function runFullSync() {
        echo "🔄 Complete Attendance System Sync\n";
        
        // Step 1: Fix Database Schema
        echo "\nStep 1: Fixing Database Schema\n";
        $this->fixDatabaseSchema();
        
        // Step 2: Check Tables
        echo "\nStep 2: Checking Database Tables\n";
        $this->checkTables();
        
        // Step 3: Show existing users
        echo "\nStep 3: Existing Users\n";
        $this->getExistingUsers();
        
        // Step 4: Test API Connection
        echo "\nStep 4: Testing API Connection\n";
        $this->testAPIConnection();
        
        // Step 5: Test Attendance Rules
        echo "\nStep 5: Attendance Rules Test\n";
        $this->testAttendanceRules();
        
        // Step 6: Fetch Users from API
        echo "\nStep 6: Fetch Users from API\n";
        $users = $this->fetchUsersFromAPI();
        
        // Step 7: Sync Users to Database
        echo "\nStep 7: Sync Users to Database\n";
        $syncResults = $this->syncUsersToDatabase($users);
        
        // Step 8: Final Report
        echo "\nStep 8: Generate Report\n";
        $this->generateSyncReport($users, $syncResults);
        
        echo "\n🚀 Sync Process Completed!\n";
    }

    public function generateSyncReport($users, $syncResults) {
        echo "\n📊 ===== SYNC REPORT =====\n";
        echo "✅ Total users from API: " . count($users) . "\n";
        echo "✅ Users synced to database: {$syncResults['success']}\n";
        echo "❌ Sync errors: {$syncResults['errors']}\n";
        
        // Count staff vs students from successfully synced data
        try {
            $stmt = $this->pdo->query("SELECT type, COUNT(*) as count FROM users GROUP BY type");
            $typeCounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $staffCount = 0;
            $studentCount = 0;
            
            foreach ($typeCounts as $row) {
                if ($row['type'] === 'staff') {
                    $staffCount = $row['count'];
                } else {
                    $studentCount = $row['count'];
                }
            }
            
            echo "👥 Staff members: $staffCount\n";
            echo "🎓 Student members: $studentCount\n";
            echo "📈 Total in database: " . ($staffCount + $studentCount) . "\n";
            
        } catch (PDOException $e) {
            echo "⚠️ Could not get user counts: " . $e->getMessage() . "\n";
        }
    }
}

// Execute the sync
try {
    date_default_timezone_set('Asia/Dhaka');
    
    echo "<pre>";
    $syncSystem = new AttendanceSystemSync();
    $syncSystem->runFullSync();
    echo "</pre>";
    
} catch (Exception $e) {
    echo "<pre>❌ System Error: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "</pre>";
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Attendance System Sync - Fixed Version</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f0f8ff; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .log { background: #1a1a1a; color: #00ff00; padding: 15px; border-radius: 5px; font-family: 'Courier New', monospace; height: 600px; overflow-y: auto; white-space: pre-wrap; }
        .btn { background: #28a745; color: white; padding: 12px 24px; border: none; border-radius: 5px; cursor: pointer; margin: 5px; font-size: 16px; }
        .btn:hover { background: #218838; }
        .btn-test { background: #17a2b8; }
        .btn-clear { background: #dc3545; }
        .status { padding: 10px; border-radius: 5px; margin: 10px 0; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .warning { background: #fff3cd; color: #856404; border: 1px solid #ffeaa7; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🏫 Attendance System Sync - Debug Version</h1>
        <div class="status success">
            <strong>Last sync:</strong> <?php echo date('Y-m-d H:i:s'); ?>
        </div>
        
        <div class="btn-group">
            <button class="btn" onclick="runSync()">🔄 Run Full Sync</button>
            <button class="btn btn-test" onclick="testAPI()">🔍 Test API Only</button>
            <button class="btn btn-clear" onclick="clearLog()">🗑️ Clear Log</button>
        </div>
        
        <h3>Sync Log:</h3>
        <div class="log" id="log">
            <?php 
            if (isset($syncSystem)) {
                echo "Sync completed at: " . date('Y-m-d H:i:s') . "\n";
                echo "Check the detailed output above for results.";
            }
            ?>
        </div>
    </div>

    <script>
        function runSync() {
            const log = document.getElementById('log');
            log.innerHTML = '🔄 Starting full synchronization...\n\n';
            log.scrollTop = log.scrollHeight;
            
            fetch(window.location.href + '?action=sync', {cache: 'no-store'})
                .then(response => response.text())
                .then(data => {
                    // Extract the pre content
                    const preContent = data.match(/<pre>([\s\S]*?)<\/pre>/);
                    if (preContent && preContent[1]) {
                        log.innerHTML = preContent[1];
                    } else {
                        log.innerHTML += '✅ Sync completed - check console for details\n';
                    }
                    log.scrollTop = log.scrollHeight;
                })
                .catch(error => {
                    log.innerHTML += '\n❌ Error: ' + error;
                    log.scrollTop = log.scrollHeight;
                });
        }
        
        function testAPI() {
            const log = document.getElementById('log');
            log.innerHTML = '🔍 Testing API connection only...\n\n';
            
            fetch(window.location.href + '?action=test', {cache: 'no-store'})
                .then(response => response.text())
                .then(data => {
                    log.innerHTML += 'API Test completed - check console\n';
                    log.scrollTop = log.scrollHeight;
                })
                .catch(error => {
                    log.innerHTML += '\n❌ API Test error: ' + error;
                    log.scrollTop = log.scrollHeight;
                });
        }
        
        function clearLog() {
            document.getElementById('log').innerHTML = 'Log cleared at: ' + new Date().toLocaleString() + '\n';
        }
    </script>
</body>
</html>