<?php
// ams_at.php - Working Attendance System with ZKTeco Simulation
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Database Configuration - UPDATE THESE WITH YOUR CREDENTIALS
define('DB_HOST', 'localhost');
define('DB_NAME', 'onesqeif_799');
define('DB_USER', 'onesqeif_799'); 
define('DB_PASS', '!QJ{xhY5Dkz1');

// Attendance Rules
define('PRESENT_DEADLINE', '08:30:00');
define('LATE_DEADLINE', '09:00:00');
define('HALF_DAY_DEADLINE', '10:30:00');

class WorkingAttendanceSystem {
    private $conn;
    private $isConnected = false;
    
    public function __construct() {
        $this->initializeDatabase();
    }
    
    private function initializeDatabase() {
        try {
            $this->conn = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ]
            );
            $this->isConnected = true;
            $this->ensureTables();
        } catch(PDOException $e) {
            $this->isConnected = false;
        }
    }
    
    private function ensureTables() {
        if (!$this->isConnected) return;
        
        $tables = [
            'users' => "CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id VARCHAR(50) UNIQUE NOT NULL,
                name VARCHAR(255) NOT NULL,
                phone VARCHAR(20),
                type ENUM('staff', 'student') NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",
            
            'attendance' => "CREATE TABLE IF NOT EXISTS attendance (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id VARCHAR(50) NOT NULL,
                check_in DATETIME NOT NULL,
                check_out DATETIME,
                status ENUM('P', 'L', 'HD', 'A') NOT NULL,
                hours_worked DECIMAL(4,2),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_user_id (user_id),
                INDEX idx_date (check_in)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",
            
            'zk_device_logs' => "CREATE TABLE IF NOT EXISTS zk_device_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                device_ip VARCHAR(20),
                user_id VARCHAR(50),
                punch_time DATETIME,
                punch_type INT,
                synced BOOLEAN DEFAULT FALSE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4"
        ];
        
        foreach ($tables as $sql) {
            try {
                $this->conn->exec($sql);
            } catch (Exception $e) {
                // Table creation error
            }
        }
    }
    
    public function isDatabaseConnected() {
        return $this->isConnected;
    }
    
    // Simulate ZKTeco device data - This is where real device integration would go
    private function simulateZKDeviceData() {
        $today = date('Y-m-d');
        
        // Simulated device punches for today
        $devicePunches = [
            // Staff punches
            ['user_id' => 'staff_1', 'punch_time' => $today . ' 08:25:00', 'punch_type' => 0],
            ['user_id' => 'staff_1', 'punch_time' => $today . ' 12:05:00', 'punch_type' => 1],
            ['user_id' => 'staff_2', 'punch_time' => $today . ' 08:45:00', 'punch_type' => 0],
            ['user_id' => 'staff_2', 'punch_time' => $today . ' 12:10:00', 'punch_type' => 1],
            
            // Student punches
            ['user_id' => 'student_101', 'punch_time' => $today . ' 08:15:00', 'punch_type' => 0],
            ['user_id' => 'student_101', 'punch_time' => $today . ' 11:55:00', 'punch_type' => 1],
            ['user_id' => 'student_102', 'punch_time' => $today . ' 09:05:00', 'punch_type' => 0],
            ['user_id' => 'student_102', 'punch_time' => $today . ' 12:00:00', 'punch_type' => 1],
            ['user_id' => 'student_103', 'punch_time' => $today . ' 10:45:00', 'punch_type' => 0],
            ['user_id' => 'student_103', 'punch_time' => $today . ' 12:15:00', 'punch_type' => 1],
        ];
        
        return $devicePunches;
    }
    
    // Process attendance status based on check-in time
    private function calculateAttendanceStatus($checkInTime) {
        $checkTime = date('H:i:s', strtotime($checkInTime));
        
        if ($checkTime <= PRESENT_DEADLINE) {
            return 'P'; // Present
        } elseif ($checkTime <= LATE_DEADLINE) {
            return 'L'; // Late
        } elseif ($checkTime <= HALF_DAY_DEADLINE) {
            return 'HD'; // Half Day
        } else {
            return 'A'; // Absent
        }
    }
    
    // Calculate hours worked
    private function calculateHoursWorked($checkIn, $checkOut) {
        if (!$checkOut) return null;
        
        $diff = strtotime($checkOut) - strtotime($checkIn);
        return round($diff / 3600, 2);
    }
    
    // Main function: Sync attendance from simulated ZKTeco device to cPanel
    public function syncAttendanceFromDevice() {
        if (!$this->isConnected) {
            return [$this->logMessage("❌ Database not connected", "error")];
        }
        
        $logs = [];
        $logs[] = $this->logMessage("🔄 Starting Attendance Sync from ZKTeco Device", "info");
        
        // Step 1: Get simulated device data
        $devicePunches = $this->simulateZKDeviceData();
        $logs[] = $this->logMessage("📥 Simulated device data: " . count($devicePunches) . " punch records", "info");
        
        // Step 2: Process punches into attendance sessions
        $attendanceSessions = $this->processPunchesToSessions($devicePunches);
        $logs[] = $this->logMessage("🕐 Created " . count($attendanceSessions) . " attendance sessions", "info");
        
        // Step 3: Save to attendance table
        $savedCount = 0;
        foreach ($attendanceSessions as $session) {
            $result = $this->saveAttendanceSession($session);
            if ($result['success']) {
                $savedCount++;
                $logs[] = $this->logMessage($result['message'], "success");
            } else {
                $logs[] = $this->logMessage($result['message'], "error");
            }
        }
        
        $logs[] = $this->logMessage("🎯 Sync completed: $savedCount attendance records saved", "success");
        
        return $logs;
    }
    
    // Process raw punches into check-in/check-out sessions
    private function processPunchesToSessions($punches) {
        $sessions = [];
        $userLastPunch = [];
        
        foreach ($punches as $punch) {
            $userId = $punch['user_id'];
            $punchTime = $punch['punch_time'];
            $isCheckOut = $punch['punch_type'] == 1;
            
            if (!$isCheckOut) {
                // Check-in punch
                $userLastPunch[$userId] = [
                    'check_in' => $punchTime,
                    'check_out' => null
                ];
            } else {
                // Check-out punch - complete the session
                if (isset($userLastPunch[$userId])) {
                    $userLastPunch[$userId]['check_out'] = $punchTime;
                    $sessions[] = [
                        'user_id' => $userId,
                        'check_in' => $userLastPunch[$userId]['check_in'],
                        'check_out' => $punchTime
                    ];
                    unset($userLastPunch[$userId]);
                }
            }
        }
        
        // Add incomplete sessions (only check-in)
        foreach ($userLastPunch as $userId => $session) {
            if ($session['check_out'] === null) {
                $sessions[] = [
                    'user_id' => $userId,
                    'check_in' => $session['check_in'],
                    'check_out' => null
                ];
            }
        }
        
        return $sessions;
    }
    
    // Save attendance session to database
    private function saveAttendanceSession($session) {
        try {
            // Check if attendance already exists for this user and date
            $stmt = $this->conn->prepare("
                SELECT id FROM attendance 
                WHERE user_id = ? AND DATE(check_in) = DATE(?) 
                LIMIT 1
            ");
            $stmt->execute([$session['user_id'], $session['check_in']]);
            
            if ($stmt->rowCount() > 0) {
                return [
                    'success' => false,
                    'message' => "⚠️ Attendance already exists for {$session['user_id']} on " . date('Y-m-d', strtotime($session['check_in']))
                ];
            }
            
            // Calculate status and hours
            $status = $this->calculateAttendanceStatus($session['check_in']);
            $hoursWorked = $this->calculateHoursWorked($session['check_in'], $session['check_out']);
            
            // Insert new attendance record
            $stmt = $this->conn->prepare("
                INSERT INTO attendance (user_id, check_in, check_out, status, hours_worked) 
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $session['user_id'],
                $session['check_in'],
                $session['check_out'],
                $status,
                $hoursWorked
            ]);
            
            return [
                'success' => true,
                'message' => "✅ {$session['user_id']}: " . 
                            date('H:i', strtotime($session['check_in'])) . " → " . 
                            "<span class='status-$status'>$status</span> " .
                            "(" . ($hoursWorked ?: 'N/A') . "h)"
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => "❌ Error saving attendance for {$session['user_id']}: " . $e->getMessage()
            ];
        }
    }
    
    // Add sample users to database
    public function addSampleUsers() {
        if (!$this->isConnected) {
            return [$this->logMessage("❌ Database not connected", "error")];
        }
        
        $sampleUsers = [
            ['staff_1', 'Mst. Parul Akter', 'staff'],
            ['staff_2', 'Mst. Rima Akter', 'staff'],
            ['student_101', 'Gazi Muhab Bin Ferdous', 'student'],
            ['student_102', 'Jamila Akter', 'student'],
            ['student_103', 'Abu Bokkar Siddik', 'student'],
            ['student_104', 'Md. Tawfiq', 'student'],
            ['student_105', 'Nusrat Jahan Sowrna', 'student'],
        ];
        
        $logs = [];
        $addedCount = 0;
        
        foreach ($sampleUsers as $user) {
            try {
                $stmt = $this->conn->prepare("
                    INSERT IGNORE INTO users (user_id, name, type) 
                    VALUES (?, ?, ?)
                ");
                $stmt->execute($user);
                
                if ($stmt->rowCount() > 0) {
                    $addedCount++;
                    $logs[] = $this->logMessage("✅ Added: {$user[1]} ({$user[2]})", "success");
                }
            } catch (Exception $e) {
                $logs[] = $this->logMessage("❌ Error adding {$user[1]}: " . $e->getMessage(), "error");
            }
        }
        
        array_unshift($logs, $this->logMessage("👥 Added $addedCount sample users", "info"));
        return $logs;
    }
    
    // Get dashboard statistics
    public function getDashboardStats() {
        if (!$this->isConnected) {
            return ['error' => 'Database not connected'];
        }
        
        try {
            $stats = [];
            
            // User statistics
            $stmt = $this->conn->query("SELECT COUNT(*) as total FROM users");
            $stats['total_users'] = $stmt->fetch()['total'];
            
            // Attendance statistics
            $stmt = $this->conn->query("SELECT COUNT(*) as total FROM attendance");
            $stats['total_attendance'] = $stmt->fetch()['total'];
            
            $stmt = $this->conn->query("SELECT COUNT(*) as today FROM attendance WHERE DATE(check_in) = CURDATE()");
            $stats['today_attendance'] = $stmt->fetch()['today'];
            
            // Today's attendance breakdown
            $stmt = $this->conn->query("
                SELECT status, COUNT(*) as count 
                FROM attendance 
                WHERE DATE(check_in) = CURDATE() 
                GROUP BY status
            ");
            $stats['today_breakdown'] = $stmt->fetchAll();
            
            // Recent attendance
            $stmt = $this->conn->query("
                SELECT a.*, u.name 
                FROM attendance a 
                LEFT JOIN users u ON a.user_id = u.user_id 
                ORDER BY a.check_in DESC 
                LIMIT 5
            ");
            $stats['recent_attendance'] = $stmt->fetchAll();
            
            return $stats;
            
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
    
    private function logMessage($message, $type = "info") {
        return "<div class='log-$type'>$message</div>";
    }
}

// HTML Interface
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AMS - Working Attendance System</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #2c3e50, #3498db);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .control-panel {
            background: #f8f9fa;
            padding: 20px;
            border-bottom: 1px solid #dee2e6;
        }
        .button-group {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            justify-content: center;
        }
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-block;
        }
        .btn-primary { background: #007bff; color: white; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: #212529; }
        .btn-info { background: #17a2b8; color: white; }
        .btn:hover { transform: translateY(-2px); box-shadow: 0 5px 15px rgba(0,0,0,0.2); }
        .content { padding: 30px; }
        .log-success { background: #d4edda; color: #155724; padding: 15px; margin: 10px 0; border-left: 5px solid #28a745; border-radius: 5px; }
        .log-error { background: #f8d7da; color: #721c24; padding: 15px; margin: 10px 0; border-left: 5px solid #dc3545; border-radius: 5px; }
        .log-warning { background: #fff3cd; color: #856404; padding: 15px; margin: 10px 0; border-left: 5px solid #ffc107; border-radius: 5px; }
        .log-info { background: #d1ecf1; color: #0c5460; padding: 15px; margin: 10px 0; border-left: 5px solid #17a2b8; border-radius: 5px; }
        .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 30px 0; }
        .stat-card { background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 25px; border-radius: 10px; text-align: center; }
        .stat-number { font-size: 2.5em; font-weight: bold; margin: 10px 0; }
        .status-P { color: #28a745; font-weight: bold; }
        .status-L { color: #ffc107; font-weight: bold; }
        .status-HD { color: #fd7e14; font-weight: bold; }
        .status-A { color: #dc3545; font-weight: bold; }
        table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background: #343a40; color: white; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🏢 Working Attendance System</h1>
            <p>ZKTeco Device Simulation - Real Data Sync</p>
        </div>
        
        <?php
        $system = new WorkingAttendanceSystem();
        $dbConnected = $system->isDatabaseConnected();
        
        if (!$dbConnected) {
            echo "<div class='log-error'>❌ Database connection failed. Please check your database credentials at the top of the file.</div>";
        } else {
            echo "<div class='log-success'>✅ Database connected successfully</div>";
        }
        ?>
        
        <div class="control-panel">
            <div class="button-group">
                <a href="?action=add_users" class="btn btn-success">👥 Add Sample Users</a>
                <a href="?action=sync_attendance" class="btn btn-primary">📥 Sync Attendance (ZK → cPanel)</a>
                <a href="?action=view_dashboard" class="btn btn-info">📊 View Dashboard</a>
                <a href="?action=clear_data" class="btn btn-warning">🗑️ Clear Test Data</a>
            </div>
        </div>
        
        <div class="content">
            <?php
            if (isset($_GET['action'])) {
                switch ($_GET['action']) {
                    case 'add_users':
                        if ($dbConnected) {
                            $logs = $system->addSampleUsers();
                            foreach ($logs as $log) {
                                echo $log;
                            }
                        }
                        break;
                        
                    case 'sync_attendance':
                        if ($dbConnected) {
                            $logs = $system->syncAttendanceFromDevice();
                            foreach ($logs as $log) {
                                echo $log;
                            }
                        }
                        break;
                        
                    case 'clear_data':
                        if ($dbConnected) {
                            try {
                                $system->conn->exec("DELETE FROM attendance");
                                $system->conn->exec("DELETE FROM users");
                                echo "<div class='log-success'>✅ All test data cleared</div>";
                            } catch (Exception $e) {
                                echo "<div class='log-error'>❌ Error clearing data: " . $e->getMessage() . "</div>";
                            }
                        }
                        break;
                        
                    case 'view_dashboard':
                    default:
                        if ($dbConnected) {
                            $stats = $system->getDashboardStats();
                            if (!isset($stats['error'])) {
                                echo "<div class='stats-grid'>";
                                echo "<div class='stat-card'><h3>Total Users</h3><div class='stat-number'>{$stats['total_users']}</div></div>";
                                echo "<div class='stat-card'><h3>Total Attendance</h3><div class='stat-number'>{$stats['total_attendance']}</div></div>";
                                echo "<div class='stat-card'><h3>Today's Records</h3><div class='stat-number'>{$stats['today_attendance']}</div></div>";
                                echo "</div>";
                                
                                if (!empty($stats['today_breakdown'])) {
                                    echo "<h3>Today's Attendance Breakdown</h3>";
                                    echo "<table>";
                                    echo "<tr><th>Status</th><th>Count</th></tr>";
                                    foreach ($stats['today_breakdown'] as $row) {
                                        echo "<tr><td><span class='status-{$row['status']}'>{$row['status']}</span></td><td>{$row['count']}</td></tr>";
                                    }
                                    echo "</table>";
                                }
                                
                                if (!empty($stats['recent_attendance'])) {
                                    echo "<h3>Recent Attendance</h3>";
                                    echo "<table>";
                                    echo "<tr><th>Name</th><th>Check-in</th><th>Status</th><th>Hours</th></tr>";
                                    foreach ($stats['recent_attendance'] as $record) {
                                        echo "<tr>
                                            <td>{$record['name']}</td>
                                            <td>" . date('H:i', strtotime($record['check_in'])) . "</td>
                                            <td><span class='status-{$record['status']}'>{$record['status']}</span></td>
                                            <td>" . ($record['hours_worked'] ?: 'N/A') . "</td>
                                        </tr>";
                                    }
                                    echo "</table>";
                                }
                            }
                        }
                        break;
                }
            }
            ?>
        </div>
    </div>
</body>
</html>